# pokerslot.tcl version 1.2
# coded by arfer <arfer.minute@gmail.com>
# on dalnet #windrop #eggdrops #tcl #eggdrop.conf #tclscript

# this script emulates a poker slot machine using the parsing engine parsecards.tcl ...
# ... to identify the best 5 card poker hand from the 5 or 6 cards dealt
# the player is initially awarded a number of chips (preset in the configuration section)
# the player is repeatedly dealt 5 or 6 cards (also preset in the configuration ...
# ... section) and on each occasion is permitted to hold any number, all or none of them
# the cards that are not held are replaced with new random cards and the string of ...
# ... cards is then sent to the parsing engine parsecards.tcl
# the best 5 card poker hand is returned, together with it's plain english poker ...
# ... name and the player is rewarded if the minimum win threshold is met or exceeded
# winnings per hand are variable depending on how good a poker hand is found but ...
# ... would need to be 2 hand rankings (or more) above the win threshold to return ...
# ... winnings of more than the basic stake
# the minimum win threshold is adjusted according to whether the player is dealt 5 or ...
# ... 6 cards
# the game continues until the player quits or has insufficient chips left to place ...
# ... another bet (chips per bet is also preset in the configuration section)

# ***** ENSURE parsecards.tcl is loaded BEFORE this script *****

# ensure channel is not +c because the script returns colorized text
# make whatever changes you wish to the configuration section
# do make an effort to properly read the usage section
# see http://www.pagat.com/vying/pokerrank.html for an explanation of poker hand rankings

# ++++++++++ CHANGELOG ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ #

# version 1.0, tested july 2005 with windrop 1.6.17 handlen32 on dalnet using ...
# ... mIRC v6.16 client

# version 1.1, modified PART, NICK & MSG bind so they only fire off is game if in play

# version 1.2 got rid of the unnecessary namespace import statement

# ++++++++++ WARRANTY +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ #

# this code comes with absolutely no warranty

# this program is free software; you can redistribute it and/or modify it under the ...
# ... terms of the gnu (general public license) as published by the free software ...
# ... foundation; either version 2 of the license, or (at your option) any later version

# this program is distributed in the hope that it will be useful, but without any ...
# ... warranty; without even the implied warranty of merchantability or fitness for a ...
# ... particular purpose

# see the gnu for more details (http://www.gnu.org/copyleft/gpl.html)

# ++++++++++ USAGE ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ #

# 1. type '.play' to start the game (assuming '.' is the trigger word prefix)
# the player is acknowledged and awarded however many starting chips is preset in ...
# ... the configuration section then the player is advised of minimum hand required ...
# ... to win

# 2. type '.deal' to deal a hand
# the number of cards preset in the configuration section (5 or 6) is dealt randomly ...
# ... from a single pack of cards

# 3. type '.hold <options>' to hold all, none or any number of the cards
# '.hold 0' will hold none (replace all) of the cards
# '.hold all' will hold all (replace none) of the cards
# '.hold c1 c2 etc' will hold the card numbers referrenced by c1 c2 etc starting at ...
# 1 for the first card, 2 for the second etc
# for example '.hold 1 3 4' (note they are separated by spaces) will hold the first, ...
# ... third and fourth cards and replace the others
# in the case of '.hold c1 c2 etc' the actual cards held/replaced are then shown
# the final cards are then shown (including any random replacement cards that were ...
# ... required) the best hand is then shown together with the chips won (if it ...
# ... represented a winning hand) together with the player's remaining chips
# if the player has insufficient chips for another deal, the game ends

# 4. type '.deal' for another hand etc etc etc

# type '.quit' at any time to end the game
# in case the current player goes afk or is split from the bot's channel without ...
# ... quitting the game, thus preventing others users from playing, any bot operator ...
# ... (with flag +o) can forcibly reset the game using '.reset'
# if the current player parts the channel then the game will reset automatically
# if the current player changes nick then the script will properly interpret the change

# trials indicate that the player has a 1 - 2% edge for both 5 and 6 cards dealt ...
# ... which is somewhat more generous than a real slot machine

# example
# (nick): .play
# (botnick): *************** Poker Slot Machine ***************
# (botnick): You have 200 chips nick
# (botnick): Minimum of Two Pair, highest being Tens to win
# (nick): .deal
# (botnick): Initial Cards: Ks Th 7c 3d Ac 9s
# (nick): .hold 1 5
# (botnick): Holding Ks Ac, Discarding Th 7c 3d 9s
# (botnick): Final Cards: Ks Ac Kh 4s 6s 4d
# (botnick): Best Hand: Kh Ks 4d 4s Ac (Two Pairs, Kings And Fours)
# (botnick): Player wins 20 chips. You have 220 chips remaining nick
# (nick): .deal
# (botnick): Initial Cards: 2c 4d 3s Ad Qd 5c
# (nick): .hold 1 2 3 6
# (botnick): Holding 2c 4d 3s 5c, Discarding Ad Qd
# (botnick): Final Cards: 2c 4d 3s 5c Js 7h
# (botnick): Best Hand: Js 7h 5c 4d 3s (Jack High, Seven Kicker)
# (botnick): House wins. You have 200 chips remaining nick
# (nick): .quit
# (botnick): Game over. Thanks nick for playing

# ++++++++++ CONFIGURATION ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ #

# edit these settings where appropriate

# the trigger character you want to precede the public commands with
# note that in particular 'play' and 'quit' are common public commands that may be ...
# ... used by other scripts
# you can ensure they only invoke this script by using an alternative trigger ...
# ... character here
set pkslot_vTrigger "."

# the irc channel you want to play the game in
# can only be a single channel with the prerequisite # prefix
set pkslot_vChannel "#pokerroom"

# the number of cards dealt (can only be set to 5 or 6)
set pkslot_vCardsDealt 6

# the number of chips you would like the player to start with
set pkslot_vChipsStart 200

# the number of chips per deal (aka per bet)
set pkslot_vChipsBet 20

# ++++++++++ CODE +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ #

# do not edit below this line

set pkslot_vVersion "1.2"
set pkslot_vIsPlaying 0
set pkslot_vHasDealt 0
set pkslot_vChipsLeft $pkslot_vChipsStart

proc pkslot_pTrigger { } {
  global pkslot_vTrigger
  return $pkslot_vTrigger
}

bind PUB -|- [pkslot_pTrigger]play pkslot_pPlayGame
bind PUB -|- [pkslot_pTrigger]deal pkslot_pDealCards
bind PUB -|- [pkslot_pTrigger]hold pkslot_pHoldCards
bind PUB -|- [pkslot_pTrigger]quit pkslot_pQuitGame

bind MSG o|o reset pkslot_pForceResetGame

bind NICK - * pkslot_pNickChanged
bind PART - * pkslot_pNickParted

proc pkslot_pPlayGame {nick uhost hand chan arg} {
  global pkslot_vChannel
  global pkslot_vCardsDealt
  global pkslot_vIsPlaying
  global pkslot_vChipsLeft
  global pkslot_vCurrentPlayer
  if {[string equal -nocase $chan $pkslot_vChannel]} {
    if {$pkslot_vCardsDealt >= 5 && $pkslot_vCardsDealt <= 6} {
      if {[llength [split [string trim $arg]]] == 0} {
        if {$pkslot_vIsPlaying == 0} {
          set pkslot_vCurrentPlayer $nick
          putserv "PRIVMSG $chan :\00312*************** Poker Slot Machine ***************\003"
          putserv "PRIVMSG $chan :You have $pkslot_vChipsLeft chips \00310$pkslot_vCurrentPlayer\003"
          switch $pkslot_vCardsDealt {
            5 {putserv "PRIVMSG $chan :Minimum of One Pair of Sevens to win"}
            6 {putserv "PRIVMSG $chan :Minimum of Two Pair, highest being Tens to win"}
          }
          set pkslot_vIsPlaying 1
          return 0
        } elseif {$pkslot_vCurrentPlayer == $nick} {putserv "PRIVMSG $chan :Huh? \00310$nick\003 ... you are already playing the game"
        } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... the game is already in play with another user"}
      } else {putserv "PRIVMSG $chan :Huh? \00310$nick\003 ... argument\(s\) are not allowed with \0033[pkslot_pTrigger]play\003"}
    } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... the script has been misconfigured with the incorrect number of cards to deal"}
  } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... the game is not configured for use in this channel"}
}

proc pkslot_pDealCards {nick uhost hand chan arg} {
  global pkslot_vChannel
  global pkslot_vIsPlaying
  global pkslot_vHasDealt
  global pkslot_vCurrentPlayer
  global pkslot_vCardsLeft
  global pkslot_vCardsDealt
  global pkslot_vInitialHand
  if {[string equal -nocase $chan $pkslot_vChannel]} {
    if {$pkslot_vIsPlaying == 1} {
      if {[llength [split [string trim $arg]]] == 0} {
        if {$pkslot_vCurrentPlayer == $nick} {
          if {$pkslot_vHasDealt == 0} {
            set pkslot_vCardsLeft [pkslot_pGenerateCards]
            set pkslot_vInitialHand [pkslot_pRandomCards $pkslot_vCardsDealt]
            putserv "PRIVMSG $chan :Initial Cards\: [pkslot_pColorizeCards $pkslot_vInitialHand]"
            set pkslot_vHasDealt 1
            return 0
          } else {putserv "PRIVMSG $chan :Huh? \00310$nick\003 ... the cards have already been dealt"}
        } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... you are not recognised as the current player"}
      } else {putserv "PRIVMSG $chan :Huh? \00310$nick\003 ... argument\(s\) are not allowed with \0033[pkslot_pTrigger]deal\003"}
    } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... the game is not in play"}
  } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... the game is not configured for use in this channel"}
}

proc pkslot_pHoldCards {nick uhost hand chan arg} {
  global pkslot_vChannel
  global pkslot_vIsPlaying
  global pkslot_vHasDealt
  global pkslot_vCurrentPlayer
  global pkslot_vCardsDealt
  global pkslot_vInitialHand
  global pkslot_vFinalHand
  global pkslot_vChipsLeft
  global pkslot_vChipsBet
  if {[string equal -nocase $chan $pkslot_vChannel]} {
    if {$pkslot_vIsPlaying == 1} {
      if {$pkslot_vCurrentPlayer == $nick} {
        if {$pkslot_vHasDealt == 1} {
          if {[llength [split [string trim $arg]]] != 0} {
            if {[llength [split [string trim $arg]]] <= $pkslot_vCardsDealt} {
              set vK [pkslot_pCheckHoldArguments [string trim $arg]]
              set vL [join [lindex [split $vK] 0]]
              set vM [join [lindex [split $vK] 1]]
              if {$vL == 1} {
                if {$vM == 1} {
                  set vN 0
                  set vO [split $pkslot_vInitialHand]
                  for {set vX 0} {$vX < [llength [split [string trim $arg]]]} {incr vX} {
                    set vP [lindex [split [string trim $arg]] $vX]
                    set vQ [lindex [split $pkslot_vInitialHand] [expr {$vP - 1}]]
                    set vO [lreplace $vO [lsearch $vO $vQ] [lsearch $vO $vQ]]
                    if {$vN == 0} {
                      set vN $vQ
                    } else {
                      set vN [lappend vN $vQ]
                    }
                  }
                  if {[llength [split [string trim $arg]]] == $pkslot_vCardsDealt} {
                    set pkslot_vFinalHand [join $vN]
                  } else {
                    set pkslot_vFinalHand "[join $vN] [pkslot_pRandomCards [llength $vO]]"
                    putserv "PRIVMSG $chan :Holding\: [pkslot_pColorizeCards [join $vN]], Discarding\: [pkslot_pColorizeCards [join $vO]]"
                  }
                }
                set vR [split [nParseCards::pParseCards $pkslot_vFinalHand] "\|"]
                set vS1 [join [lindex $vR 0]]
                set vS2 [join [lindex $vR 1]]
                set vS3 [join [lindex $vR 2]]
                set vT [pkslot_pGetResult $vS3]
                putserv "PRIVMSG $chan :Final Cards\: [pkslot_pColorizeCards $pkslot_vFinalHand]"
                putserv "PRIVMSG $chan :Best Hand\: [pkslot_pColorizeCards $vS1] \($vS2\)"
                if {$vT == 0} {
                  set pkslot_vChipsLeft [expr {$pkslot_vChipsLeft - $pkslot_vChipsBet}]
                  if {$pkslot_vChipsLeft < $pkslot_vChipsBet} {
                    putserv "PRIVMSG $chan :House Wins. You have insufficient funds for another hand \00310$pkslot_vCurrentPlayer\003"
                    putserv "PRIVMSG $chan :Game over. Thanks \00310$pkslot_vCurrentPlayer\003 for playing"
                    pkslot_pResetGame
                    return
                  } else {
                    set vU "House wins."
                  }
                } else {
                  set pkslot_vChipsLeft [expr {$pkslot_vChipsLeft + int ($vT * $pkslot_vChipsBet)}]
                  set vU "Player wins [expr {int ($vT * $pkslot_vChipsBet)}] chips."
                }
                putserv "PRIVMSG $chan :$vU You have $pkslot_vChipsLeft chips remaining \00310$pkslot_vCurrentPlayer\003"
                set pkslot_vHasDealt 0
              } else {putserv "PRIVMSG $chan :Huh? \00310$nick\003 ... incorrect or duplicated argument\(s\) with \0033[pkslot_pTrigger]hold\003"}
            } else {putserv "PRIVMSG $chan :Huh? \00310$nick\003 ... too many arguments with \0033[pkslot_pTrigger]hold\003"}
          } else {putserv "PRIVMSG $chan :Huh? \00310$nick\003 ... argument\(s\) are required with \0033[pkslot_pTrigger]hold\003"}
        } else {putserv "PRIVMSG $chan :Huh? \00310$nick\003 ... the cards haven't been dealt yet"}
      } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... you are not recognised as the current player"}
    } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... the game is not in play"}
  } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... the game is not configured for use in this channel"}
}

proc pkslot_pQuitGame {nick uhost hand chan arg} {
  global pkslot_vChannel
  global pkslot_vIsPlaying
  global pkslot_vCurrentPlayer
  if {[string equal -nocase $chan $pkslot_vChannel]} {
    if {$pkslot_vIsPlaying == 1} {
      if {[llength [split [string trim $arg]]] == 0} {
        if {$pkslot_vCurrentPlayer == $nick} {
          pkslot_pResetGame
          putserv "PRIVMSG $chan :Game over. Thanks \00310$pkslot_vCurrentPlayer\003 for playing"
        } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... you are not recognised as the current player"}
      } else {putserv "PRIVMSG $chan :Huh? \00310$nick\003 ... argument\(s\) are not allowed with \0033[pkslot_pTrigger]quit\003"}
    } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... the game is not in play"}
  } else {putserv "PRIVMSG $chan :Sorry \00310$nick\003 ... the game is not configured for use in this channel"}
}

proc pkslot_pForceResetGame {nick uhost hand text} {
  global pkslot_vIsPlaying
  global pkslot_vChannel
  global pkslot_vCurrentPlayer
  if {$pkslot_vIsPlaying == 1} {
    pkslot_pResetGame
    putserv "PRIVMSG $pkslot_vChannel :Sorry \00310$pkslot_vCurrentPlayer\003 ... the game has been forcibly reset by \00310$nick\003"
  }
}

proc pkslot_pNickChanged {nick uhost hand chan newnick} {
  global pkslot_vIsPlaying
  global pkslot_vCurrentPlayer
  if {$pkslot_vIsPlaying == 1} {
    if {$nick == $pkslot_vCurrentPlayer} {
      set pkslot_vCurrentPlayer $newnick
    }
  }
}

proc pkslot_pNickParted {nick uhost hand chan {msg ""}} {
  global pkslot_vIsPlaying
  global pkslot_vCurrentPlayer
  global pkslot_vChannel
  if {$pkslot_vIsPlaying == 1} {
    if {$nick == $pkslot_vCurrentPlayer && [string equal -nocase $chan $pkslot_vChannel]} {
      pkslot_pResetGame
      putserv "PRIVMSG $chan : Game over. \00310$nick\003 has parted \0035$pkslot_vChannel\003"
    }
  }
}

proc pkslot_pCheckHoldArguments {vA} {
  global pkslot_vInitialHand
  global pkslot_vFinalHand
  global pkslot_vCardsDealt
  set vK 1
  set vL 0
  if {$vA == "all"} {
    set pkslot_vFinalHand $pkslot_vInitialHand
  } elseif {$vA == 0} {
    set pkslot_vFinalHand [pkslot_pRandomCards $pkslot_vCardsDealt]
  } else {
    set vL 1
    for {set vX 0} {$vX < [llength [split $vA]]} {incr vX} {
      set vM [lindex [split $vA] $vX]
      if {![regexp {^[1-6]$} [join $vM]]} {
        set vK 0
        break
      } else {
        if {$vM > $pkslot_vCardsDealt || [llength [lsearch -all -inline [split $vA] $vM]] > 1} {
          set vK 0
          break
        }
      }
    }
  }
  return "$vK $vL"
}

proc pkslot_pGetResult {vA} {
  global pkslot_vCardsDealt
  global pkslot_vChipsBet
  set vK 0
  switch $pkslot_vCardsDealt {
    5 {
      if {[lindex [split $vA] 0] == 9 && [lindex [split $vA] 1] <= 8} {
        set vK 1
      } else {
        switch [join [lindex [split $vA] 0]] {
          8 {set vK 1}
          7 {set vK 1.25}
          6 {set vK 1.5}
          5 {set vK 1.75}
          4 {set vK 2}
          3 {set vK 2.25}
          2 {set vK 2.5}
          1 {set vK 2.75}
        }
      }
    }
    6 {
      if {[lindex [split $vA] 0] == 8 && [lindex [split $vA] 1] <= 5} {
        set vK 1
      } else {
        switch [join [lindex [split $vA] 0]] {
          7 {set vK 1}
          6 {set vK 1.25}
          5 {set vK 1.5}
          4 {set vK 1.75}
          3 {set vK 2}
          2 {set vK 2.25}
          1 {set vK 2.5}
        }
      }
    }
  }
  return $vK
}

proc pkslot_pGenerateCards {} {
  set vK "AKQJT98765432"
  set vL "hcds"
  set vM 0
  for {set vX 0} {$vX < 4} {incr vX} {
    for {set vY 0} {$vY < 13} {incr vY} {
      set vN "[string index $vK $vY][string index $vL $vX]"
      if {$vM == 0} {set vM $vN} else {set vM "$vM $vN"}
    }
  }
  return $vM
}

proc pkslot_pRandomCards {vA} {
  global pkslot_vCardsLeft
  set vL 0
  for {set vX 1} {$vX <= $vA} {incr vX} {
    set vM [rand [llength [split $pkslot_vCardsLeft]]]
    set vN [join [lindex [split $pkslot_vCardsLeft] $vM]]
    if {$vL == 0} {set vL $vN} else {set vL "$vL $vN"}
    set pkslot_vCardsLeft [join [lreplace [split $pkslot_vCardsLeft] $vM $vM]]
  }
  return $vL
}

proc pkslot_pColorizeCards {vA} {
  set vK 0
  for {set vX 0} {$vX < [llength [split $vA]]} {incr vX} {
    set vL [join [lindex [split $vA] $vX]]
    if {[string index $vL 1] == "h" || [string index $vL 1] == "d"} {set vL "\00304$vL\003"} else {set vL "\00314$vL\003"}
    if {$vK == 0} {set vK $vL} else {set vK "$vK $vL"}
  }
  return $vK
}

proc pkslot_pResetGame {} {
  global pkslot_vIsPlaying
  global pkslot_vHasDealt
  global pkslot_vChipsLeft
  global pkslot_vChipsStart
  set pkslot_vIsPlaying 0
  set pkslot_vHasDealt 0
  set pkslot_vChipsLeft $pkslot_vChipsStart
}

putlog "pokerslot.tcl version $pkslot_vVersion by arfer loaded"
